/**
 * Storage Manager — Local file storage for recordings and snapshots.
 *
 * Ported from storage_manager.py.
 *
 * Directory structure:
 *   ~/.ami/users/{userId}/recordings/{sessionId}/
 *     recording.json    (metadata + operations)
 *     snapshots/
 *       {urlHash}.json  (page snapshots)
 */
import { mkdirSync, existsSync, rmSync, readdirSync, readFileSync, writeFileSync, } from "node:fs";
import { join } from "node:path";
import { AMI_DIR } from "../utils/config.js";
import { createLogger } from "../utils/logging.js";
const logger = createLogger("storage-manager");
// ===== Storage Manager =====
export class StorageManager {
    basePath;
    constructor(basePath) {
        this.basePath = basePath ?? AMI_DIR;
    }
    // ===== Path Helpers =====
    sanitizeId(id) {
        return id.replace(/[/\\]/g, "_").replace(/\.\./g, "_");
    }
    userPath(userId) {
        const dir = join(this.basePath, "users", this.sanitizeId(userId));
        mkdirSync(dir, { recursive: true });
        return dir;
    }
    recordingPath(userId, sessionId) {
        return join(this.userPath(userId), "recordings", this.sanitizeId(sessionId));
    }
    recordingFile(userId, sessionId) {
        return join(this.recordingPath(userId, sessionId), "recording.json");
    }
    snapshotsDir(userId, sessionId) {
        return join(this.recordingPath(userId, sessionId), "snapshots");
    }
    // ===== Recording CRUD =====
    saveRecording(userId, sessionId, data, updateTimestamp = true) {
        const dir = this.recordingPath(userId, sessionId);
        mkdirSync(dir, { recursive: true });
        if (updateTimestamp) {
            data.updated_at = new Date().toISOString();
        }
        const filepath = this.recordingFile(userId, sessionId);
        writeFileSync(filepath, JSON.stringify(data, null, 2), "utf-8");
        logger.debug({ filepath }, "Recording saved");
    }
    saveSnapshot(userId, sessionId, urlHash, snapshotData) {
        const dir = this.snapshotsDir(userId, sessionId);
        mkdirSync(dir, { recursive: true });
        const filepath = join(dir, `${urlHash}.json`);
        writeFileSync(filepath, JSON.stringify(snapshotData, null, 2), "utf-8");
    }
    getRecording(userId, sessionId) {
        const filepath = this.recordingFile(userId, sessionId);
        if (!existsSync(filepath))
            return null;
        try {
            const data = JSON.parse(readFileSync(filepath, "utf-8"));
            // Load snapshots
            const snapshotsDir = this.snapshotsDir(userId, sessionId);
            if (existsSync(snapshotsDir)) {
                const snapshots = {};
                for (const file of readdirSync(snapshotsDir)) {
                    if (file.endsWith(".json")) {
                        const urlHash = file.replace(".json", "");
                        try {
                            snapshots[urlHash] = JSON.parse(readFileSync(join(snapshotsDir, file), "utf-8"));
                        }
                        catch (snapErr) {
                            logger.warn({ file, err: snapErr }, "Skipping corrupted snapshot file");
                        }
                    }
                }
                if (Object.keys(snapshots).length > 0) {
                    data.snapshots = snapshots;
                }
            }
            return data;
        }
        catch (err) {
            logger.error({ err, filepath }, "Failed to read recording");
            return null;
        }
    }
    listRecordings(userId) {
        const recordingsDir = join(this.userPath(userId), "recordings");
        if (!existsSync(recordingsDir))
            return [];
        const items = [];
        for (const sessionId of readdirSync(recordingsDir)) {
            const filepath = this.recordingFile(userId, sessionId);
            if (!existsSync(filepath))
                continue;
            try {
                const data = JSON.parse(readFileSync(filepath, "utf-8"));
                const snapshotsDir = this.snapshotsDir(userId, sessionId);
                const snapshotCount = existsSync(snapshotsDir)
                    ? readdirSync(snapshotsDir).filter((f) => f.endsWith(".json")).length
                    : 0;
                items.push({
                    session_id: sessionId,
                    task_metadata: data.task_metadata,
                    created_at: data.created_at,
                    action_count: data.operations?.length ?? 0,
                    snapshot_count: snapshotCount,
                });
            }
            catch {
                // Skip invalid recordings
            }
        }
        // Sort by created_at descending (newest first)
        items.sort((a, b) => (b.created_at ?? "").localeCompare(a.created_at ?? ""));
        return items;
    }
    deleteRecording(userId, sessionId) {
        const dir = this.recordingPath(userId, sessionId);
        if (!existsSync(dir))
            return false;
        try {
            rmSync(dir, { recursive: true, force: true });
            logger.info({ userId, sessionId }, "Recording deleted");
            return true;
        }
        catch (err) {
            logger.error({ err, dir }, "Failed to delete recording");
            return false;
        }
    }
    getRecordingDetail(userId, sessionId) {
        const data = this.getRecording(userId, sessionId);
        if (!data)
            return null;
        return {
            session_id: sessionId,
            created_at: data.created_at,
            updated_at: data.updated_at,
            action_count: data.operations?.length ?? 0,
            task_metadata: data.task_metadata,
            operations: data.operations,
            snapshots: data.snapshots,
        };
    }
    // ===== Metadata Updates =====
    updateRecordingMetadata(userId, sessionId, updates) {
        const data = this.getRecording(userId, sessionId);
        if (!data)
            return false;
        const metadata = (data.task_metadata ?? {});
        if (updates.name)
            metadata.name = updates.name;
        if (updates.task_description)
            metadata.description = updates.task_description;
        if (updates.user_query)
            metadata.user_query = updates.user_query;
        data.task_metadata = metadata;
        // Remove snapshots from save (they're in separate files)
        const { snapshots: _, ...saveData } = data;
        this.saveRecording(userId, sessionId, saveData);
        return true;
    }
}
// ===== Singleton =====
let _storageManager = null;
export function getStorageManager() {
    if (!_storageManager) {
        _storageManager = new StorageManager();
    }
    return _storageManager;
}
//# sourceMappingURL=storage-manager.js.map