/**
 * BehaviorRecorder — Records user behavior using ref-based element identification.
 *
 * Ported from behavior_recorder.py.
 *
 * Key concepts:
 * - CDP session per tab for JS → Node binding
 * - Injects behavior_tracker.js via Page.addScriptToEvaluateOnNewDocument
 * - Handles Runtime.bindingCalled for click/type/scroll/navigate events
 * - Network response monitoring for dataload detection
 * - Navigation deduplication (2s window)
 * - Auto-hooks new tab creation
 */
import { readFileSync } from "node:fs";
import { resolve, dirname } from "node:path";
import { fileURLToPath } from "node:url";
import { createLogger } from "../utils/logging.js";
const logger = createLogger("behavior-recorder");
// ===== Tracker script cache =====
let _trackerJsCache = null;
function getTrackerScript() {
    if (_trackerJsCache !== null)
        return _trackerJsCache;
    const __filename = fileURLToPath(import.meta.url);
    const __dirname = dirname(__filename);
    // behavior_tracker.js lives alongside this file in scripts/
    const candidates = [
        resolve(__dirname, "scripts/behavior_tracker.js"),
        // Fallback: relative to daemon-ts/dist/browser/ at runtime
        resolve(__dirname, "../src/browser/scripts/behavior_tracker.js"),
    ];
    for (const candidate of candidates) {
        try {
            _trackerJsCache = readFileSync(candidate, "utf-8");
            logger.info({ path: candidate }, "Loaded behavior_tracker.js");
            return _trackerJsCache;
        }
        catch {
            // try next
        }
    }
    // Minimal fallback
    logger.warn("Using minimal fallback tracker script");
    _trackerJsCache = `
(function() {
  if (window._behaviorTrackerInitialized) return;
  window._behaviorTrackerInitialized = true;
  console.log("Behavior Tracker (fallback) initialized");
  function report(type, data) {
    if (window.reportUserBehavior) {
      const payload = { type, timestamp: new Date().toISOString(), url: location.href, ...data };
      window.reportUserBehavior(JSON.stringify(payload));
    }
  }
  document.addEventListener('click', e => {
    const ref = e.target.getAttribute('aria-ref');
    if (ref) report('click', { ref, text: e.target.textContent?.slice(0, 100) });
  }, true);
})();
`;
    return _trackerJsCache;
}
// ===== BehaviorRecorder class =====
export class BehaviorRecorder {
    sessionId;
    _isRecording = false;
    _enableSnapshotCapture;
    // Operation storage
    operations = [];
    snapshots = {};
    // Tab tracking
    _monitoredTabs = new Set();
    _tabPages = new Map();
    _cdpSessions = new Map();
    _fallbackTabCounter = 0;
    // Navigation deduplication
    _lastNavUrl = null;
    _lastNavTime = null;
    _navDedupMs = 2000;
    // Browser session reference
    _browserSession = null;
    // Operation callback
    _operationCallback = null;
    // Dataload detection
    _recentDataloadUrls = new Set();
    _dataloadCleanupTimer = null;
    _lastScrollTime = null;
    _dataloadWindowMs = 3000;
    // Per-page response listener refs (for targeted removal without removeAllListeners)
    _responseListeners = new Map();
    constructor(enableSnapshotCapture = true) {
        this.sessionId = `session_${new Date().toISOString().replace(/[:.]/g, "").slice(0, 15)}`;
        this._enableSnapshotCapture = enableSnapshotCapture;
    }
    setOperationCallback(callback) {
        this._operationCallback = callback;
    }
    isRecording() {
        return this._isRecording;
    }
    getOperations() {
        return [...this.operations];
    }
    getOperationsCount() {
        return this.operations.length;
    }
    // ===== Start / Stop =====
    async startRecording(browserSession) {
        if (this._isRecording) {
            logger.warn("Recording already in progress");
            return;
        }
        this._browserSession = browserSession;
        this._isRecording = true;
        this.operations = [];
        this.snapshots = {};
        this._monitoredTabs.clear();
        this._tabPages.clear();
        this._fallbackTabCounter = 0;
        logger.info({ sessionId: this.sessionId }, "Starting behavior recording");
        // Setup recording for all existing tabs
        await this._setupAllTabs();
        // Start dataload URL cleanup interval
        this._dataloadCleanupTimer = setInterval(() => {
            this._recentDataloadUrls.clear();
        }, 10_000);
    }
    async stopRecording() {
        if (!this._isRecording) {
            logger.warn("No recording in progress");
            return { session_id: this.sessionId, operations: [], operations_count: 0, snapshots: {} };
        }
        this._isRecording = false;
        logger.info({ operationCount: this.operations.length }, "Stopping recording");
        const result = {
            session_id: this.sessionId,
            operations: [...this.operations],
            operations_count: this.operations.length,
            snapshots: { ...this.snapshots },
        };
        // Detach CDP sessions and remove page listeners
        for (const [tabId, cdpSession] of this._cdpSessions) {
            try {
                await cdpSession.detach();
            }
            catch (e) {
                logger.debug({ tabId, err: e }, "Error detaching CDP session");
            }
        }
        // Remove only our response listeners (not other subsystems' listeners)
        for (const [tabId, page] of this._tabPages) {
            try {
                const listener = this._responseListeners.get(tabId);
                if (listener) {
                    page.off("response", listener);
                }
            }
            catch {
                // page may be closed
            }
        }
        this._responseListeners.clear();
        // Cleanup
        this._monitoredTabs.clear();
        this._tabPages.clear();
        this._cdpSessions.clear();
        this._browserSession = null;
        this._recentDataloadUrls.clear();
        if (this._dataloadCleanupTimer) {
            clearInterval(this._dataloadCleanupTimer);
            this._dataloadCleanupTimer = null;
        }
        return result;
    }
    // ===== Tab Setup =====
    async _setupAllTabs() {
        if (!this._browserSession)
            return;
        const pages = this._browserSession._pages;
        for (const [tabId, page] of pages) {
            if (!this._monitoredTabs.has(tabId)) {
                await this._setupForTab(tabId, page);
            }
        }
    }
    async _setupForTab(tabId, page) {
        if (this._monitoredTabs.has(tabId))
            return;
        // Avoid duplicate setup by page identity
        for (const existingPage of this._tabPages.values()) {
            if (existingPage === page)
                return;
        }
        if (page.isClosed()) {
            logger.debug({ tabId }, "Tab is closed, skipping");
            return;
        }
        try {
            logger.debug({ tabId }, "Setting up recording for tab");
            // Create CDP session
            const cdpSession = await page.context().newCDPSession(page);
            // Enable required domains
            await cdpSession.send("Runtime.enable");
            await cdpSession.send("Page.enable");
            // Add binding for JS → Node communication
            await cdpSession.send("Runtime.addBinding", { name: "reportUserBehavior" });
            // Register event handlers
            cdpSession.on("Runtime.bindingCalled", (event) => {
                this._handleBindingEvent(event, tabId);
            });
            cdpSession.on("Page.frameNavigated", (event) => {
                this._handleNavigation(event, tabId);
            });
            // Inject tracking script on new documents
            const script = getTrackerScript();
            await cdpSession.send("Page.addScriptToEvaluateOnNewDocument", {
                source: script,
                runImmediately: true,
            });
            // Inject immediately for current page
            try {
                await page.evaluate(script);
            }
            catch (e) {
                logger.debug({ err: e }, "Could not inject script immediately");
            }
            // Setup network response listener for dataload detection
            const responseListener = (response) => {
                this._handleResponse(response, tabId);
            };
            page.on("response", responseListener);
            this._responseListeners.set(tabId, responseListener);
            this._monitoredTabs.add(tabId);
            this._tabPages.set(tabId, page);
            this._cdpSessions.set(tabId, cdpSession);
            // Record current URL as initial navigation for late-attached tabs
            this._recordInitialNavigation(tabId, page);
            logger.info({ tabId }, "Recording setup complete for tab");
        }
        catch (e) {
            logger.error({ tabId, err: e }, "Failed to setup recording for tab");
        }
    }
    _recordInitialNavigation(tabId, page) {
        try {
            const url = page.url();
            if (!url || url === "about:blank" || url.startsWith("chrome://"))
                return;
            this._handleNavigation({ frame: { url } }, tabId);
        }
        catch {
            // ignore
        }
    }
    // ===== Event Handling =====
    _handleBindingEvent(event, tabId) {
        if (event.name !== "reportUserBehavior")
            return;
        const payload = event.payload;
        this._processOperation(payload, tabId).catch((e) => {
            logger.debug({ err: e }, "Error processing binding event");
        });
    }
    async _processOperation(payload, tabId) {
        try {
            const data = JSON.parse(payload);
            if (!data.type) {
                logger.warn("Invalid operation: missing type");
                return;
            }
            // Navigation deduplication
            if (data.type === "navigate") {
                const navUrl = data.url || "";
                const now = Date.now();
                if (this._lastNavUrl && this._lastNavTime) {
                    const timeDiff = now - this._lastNavTime;
                    if (navUrl === this._lastNavUrl && timeDiff < this._navDedupMs) {
                        logger.debug({ url: navUrl }, "Duplicate navigate filtered");
                        return;
                    }
                }
                this._lastNavUrl = navUrl;
                this._lastNavTime = now;
            }
            // Track scroll time for dataload detection
            if (data.type === "scroll") {
                this._lastScrollTime = Date.now();
            }
            // Add tab_id
            data.tab_id = tabId;
            // Store operation
            this.operations.push(data);
            // Log
            this._logOperation(data);
            // Callback
            if (this._operationCallback) {
                try {
                    this._operationCallback(data);
                }
                catch (e) {
                    logger.warn({ err: e }, "Operation callback failed");
                }
            }
            // Capture snapshot for navigation
            if (this._enableSnapshotCapture && data.type === "navigate") {
                const url = data.url || "";
                if (url && url !== "about:blank" && !url.startsWith("chrome://")) {
                    this._captureSnapshot(url, tabId).catch(() => { });
                }
            }
        }
        catch (e) {
            logger.warn({ err: e }, "Failed to parse operation data");
        }
    }
    _handleNavigation(event, tabId) {
        const frame = event.frame || {};
        const url = frame.url;
        const parentId = frame.parentId;
        // Only handle main frame navigation
        if (parentId !== undefined)
            return;
        if (!url || url === "about:blank" || url.startsWith("chrome://"))
            return;
        logger.debug({ tabId, url }, "CDP navigation detected");
        const navPayload = JSON.stringify({
            type: "navigate",
            timestamp: new Date().toISOString(),
            url,
        });
        this._processOperation(navPayload, tabId).catch(() => { });
    }
    // ===== Dataload Detection =====
    _handleResponse(response, tabId) {
        if (!this._isRecording)
            return;
        this._processResponse(response, tabId).catch(() => { });
    }
    async _processResponse(response, tabId) {
        try {
            // Only record dataload after recent scroll
            if (!this._lastScrollTime)
                return;
            if (Date.now() - this._lastScrollTime > this._dataloadWindowMs)
                return;
            const request = response.request();
            const resourceType = request.resourceType();
            if (resourceType !== "xhr" && resourceType !== "fetch")
                return;
            const status = response.status();
            if (status < 200 || status >= 300)
                return;
            const contentType = (await response.allHeaders())["content-type"] || "";
            if (!contentType.includes("application/json"))
                return;
            const requestUrl = request.url();
            const urlBase = requestUrl.split("?")[0];
            if (this._recentDataloadUrls.has(urlBase))
                return;
            this._recentDataloadUrls.add(urlBase);
            const data = {
                type: "dataload",
                timestamp: new Date().toISOString(),
                url: response.frame()?.url() || "",
                request_url: requestUrl,
                method: request.method(),
                status,
                tab_id: tabId,
            };
            this.operations.push(data);
            this._logOperation(data);
            if (this._operationCallback) {
                try {
                    this._operationCallback(data);
                }
                catch (e) {
                    logger.warn({ err: e }, "Operation callback failed");
                }
            }
        }
        catch (e) {
            logger.debug({ err: e }, "Error processing response");
        }
    }
    // ===== Snapshot Capture =====
    async _captureSnapshot(url, tabId) {
        if (!this._browserSession || !this._enableSnapshotCapture)
            return;
        const { createHash } = await import("node:crypto");
        const urlHash = createHash("md5").update(url).digest("hex").slice(0, 12);
        if (this.snapshots[urlHash])
            return;
        try {
            // Wait for page to stabilize
            await new Promise((r) => setTimeout(r, 1000));
            let page = this._tabPages.get(tabId);
            if ((!page || page.isClosed()) && this._browserSession) {
                page = this._browserSession._pages.get(tabId);
            }
            if (!page || page.isClosed())
                return;
            // Use BrowserSession's snapshot if available
            if (this._browserSession.snapshot &&
                this._browserSession._currentTabId === tabId) {
                const snapshotResult = await this._browserSession.snapshot.getFullResult();
                if (snapshotResult) {
                    this.snapshots[urlHash] = {
                        url,
                        snapshot_text: snapshotResult.snapshotText,
                        captured_at: new Date().toISOString(),
                    };
                    logger.info({ url: url.slice(0, 60) }, "Snapshot captured");
                    return;
                }
            }
            // Fallback: simple extraction
            const domContent = await page.evaluate(() => ({
                title: document.title,
                url: window.location.href,
            }));
            this.snapshots[urlHash] = {
                url,
                simple: domContent,
                captured_at: new Date().toISOString(),
            };
            logger.info({ url: url.slice(0, 60) }, "Simple snapshot captured");
        }
        catch (e) {
            logger.warn({ url, err: e }, "Failed to capture snapshot");
        }
    }
    // ===== Logging =====
    _logOperation(data) {
        const opType = (data.type || "unknown").toUpperCase();
        const ref = data.ref || "";
        const text = data.text ? data.text.slice(0, 30) : "";
        const parts = [`${opType}`];
        if (ref)
            parts.push(`ref=${ref}`);
        if (text)
            parts.push(`text="${text}"`);
        if (data.value)
            parts.push(`value="${String(data.value).slice(0, 30)}"`);
        if (data.url && opType === "NAVIGATE")
            parts.push(`url=${data.url.slice(0, 50)}`);
        if (data.request_url && opType === "DATALOAD")
            parts.push(`request=${data.request_url.slice(0, 60)}`);
        logger.debug({ operation: parts.join(" ") }, "Recorded operation");
    }
}
//# sourceMappingURL=behavior-recorder.js.map