/**
 * Budget Tracker — tracks LLM cost per task.
 *
 * Ported from the budget tracking concept in the Python daemon.
 * Uses pi-ai response.usage.cost data when available.
 */
import { createLogger } from "../utils/logging.js";
const logger = createLogger("budget");
// ===== Budget Tracker =====
export class BudgetTracker {
    _totalCost = 0;
    _totalInputTokens = 0;
    _totalOutputTokens = 0;
    _callCount = 0;
    _budgetLimit;
    _taskId;
    constructor(taskId, budgetLimit = Infinity) {
        this._taskId = taskId;
        this._budgetLimit = budgetLimit;
    }
    get totalCost() {
        return this._totalCost;
    }
    get totalInputTokens() {
        return this._totalInputTokens;
    }
    get totalOutputTokens() {
        return this._totalOutputTokens;
    }
    get callCount() {
        return this._callCount;
    }
    get isOverBudget() {
        return this._totalCost >= this._budgetLimit;
    }
    /**
     * Record usage from a pi-ai response.
     * pi-ai Usage type: { input, output, cacheRead, cacheWrite, totalTokens, cost: { total, ... } }
     */
    recordUsage(usage) {
        this._callCount++;
        if (usage.input) {
            this._totalInputTokens += usage.input;
        }
        if (usage.output) {
            this._totalOutputTokens += usage.output;
        }
        const cost = usage.cost?.total ?? 0;
        this._totalCost += cost;
        logger.debug({
            taskId: this._taskId,
            callCount: this._callCount,
            inputTokens: usage.input,
            outputTokens: usage.output,
            cost,
            totalCost: this._totalCost,
        }, "Usage recorded");
        if (this.isOverBudget) {
            logger.warn({
                taskId: this._taskId,
                totalCost: this._totalCost,
                budgetLimit: this._budgetLimit,
            }, "Budget limit exceeded");
        }
    }
    /**
     * Get a summary of token usage.
     */
    getSummary() {
        return {
            totalCost: this._totalCost,
            totalInputTokens: this._totalInputTokens,
            totalOutputTokens: this._totalOutputTokens,
            callCount: this._callCount,
            isOverBudget: this.isOverBudget,
        };
    }
    /**
     * Reset all counters.
     */
    reset() {
        this._totalCost = 0;
        this._totalInputTokens = 0;
        this._totalOutputTokens = 0;
        this._callCount = 0;
    }
}
//# sourceMappingURL=budget.js.map