/**
 * TaskState — per-task state management.
 *
 * Manages event queue, human response queue, user message queue,
 * conversation history, and execution metadata.
 */
import { SSEEmitter } from "../events/emitter.js";
export declare enum TaskStatus {
    PENDING = "pending",
    RUNNING = "running",
    WAITING = "waiting",
    COMPLETED = "completed",
    FAILED = "failed",
    CANCELLED = "cancelled"
}
export interface ConversationEntry {
    role: "user" | "assistant" | "system" | "task_result" | "tool_call";
    content: string;
    timestamp: string;
}
export declare class TaskState {
    readonly taskId: string;
    task: string;
    userId?: string;
    projectId?: string;
    status: TaskStatus;
    progress: number;
    result?: unknown;
    error?: string;
    readonly createdAt: Date;
    startedAt?: Date;
    completedAt?: Date;
    updatedAt: Date;
    toolsCalled: Record<string, unknown>[];
    loopIteration: number;
    conversationHistory: ConversationEntry[];
    lastTaskResult?: string;
    subtasks: Record<string, unknown>[];
    summaryTask?: string;
    readonly emitter: SSEEmitter;
    abortController: AbortController;
    private _humanResponseQueue;
    private _userMessageQueue;
    private _userMessageWaiters;
    private _pauseDeferred?;
    constructor(taskId: string, task: string);
    markRunning(): void;
    markWaiting(): void;
    markCompleted(result?: unknown): void;
    markFailed(error: string): void;
    markCancelled(reason?: string): void;
    /** Resolve all pending deferred queues (used on cancel/cleanup). */
    private _drainWaiters;
    addConversation(role: ConversationEntry["role"], content: string): void;
    getRecentContext(maxEntries?: number): string;
    /** Wait for a human response. Resolves when provideHumanResponse() is called. */
    waitForHumanResponse(timeoutMs?: number): Promise<string | null>;
    /** Provide a human response to the waiting ask_human tool. */
    provideHumanResponse(response: string): boolean;
    /** Put a user message into the queue for multi-turn processing. */
    putUserMessage(message: string): void;
    /** Get the next user message, with optional timeout. Returns null on timeout or cancel. */
    getUserMessage(timeoutMs?: number): Promise<string | null>;
    /**
     * Cancel the most recent getUserMessage() waiter.
     * Used when an executor wins the Promise.race in waitForEvent() —
     * the losing getUserMessage() deferred must be removed so it doesn't
     * consume the next putUserMessage() call.
     */
    cancelLastGetUserMessage(): void;
    /** Check if there are pending user messages without consuming them. */
    get hasPendingUserMessages(): boolean;
    /** Pause execution. Returns a promise that resolves when resume() is called. */
    pause(): Promise<void>;
    /** Resume paused execution. No-op if already cancelled/completed/failed. */
    resume(): boolean;
    get isPaused(): boolean;
    get durationSeconds(): number | undefined;
    toJSON(): Record<string, unknown>;
}
//# sourceMappingURL=task-state.d.ts.map