/**
 * SSE Emitter — queue-based event emission for streaming to clients.
 *
 * Wire format: `data: {"step": "<action>", "data": {...}}\n\n`
 */
import { Action, } from "./types.js";
import { createLogger } from "../utils/logging.js";
const logger = createLogger("sse-emitter");
const MAX_QUEUE_SIZE = 1000;
// ===== SSE Formatting =====
export function sseJson(step, data) {
    const payload = { step, data };
    return `data: ${JSON.stringify(payload)}\n\n`;
}
export function sseAction(actionData) {
    const action = actionData.action;
    return sseJson(action, actionData);
}
export function sseComment(comment) {
    return `: ${comment}\n\n`;
}
export function sseHeartbeat() {
    return sseJson("heartbeat", {
        action: "heartbeat",
        message: "keep-alive",
        timestamp: new Date().toISOString(),
    });
}
function createDeferred() {
    let resolve;
    let reject;
    const promise = new Promise((res, rej) => {
        resolve = res;
        reject = rej;
    });
    return { resolve, reject, promise };
}
// ===== SSEEmitter Class =====
export class SSEEmitter {
    _queue = [];
    _waiters = [];
    _closed = false;
    _taskId;
    _agentName;
    _startTime;
    configure(taskId, agentName) {
        this._taskId = taskId;
        this._agentName = agentName;
        this._startTime = Date.now();
        return this;
    }
    get taskId() {
        return this._taskId;
    }
    get isClosed() {
        return this._closed;
    }
    emit(actionData) {
        if (this._closed)
            return;
        // Stamp task_id if configured and not already set
        if (this._taskId && !actionData.task_id) {
            actionData.task_id = this._taskId;
        }
        // Stamp timestamp if not set
        if (!actionData.timestamp) {
            actionData.timestamp = new Date().toISOString();
        }
        // If someone is waiting, resolve immediately
        if (this._waiters.length > 0) {
            const waiter = this._waiters.shift();
            try {
                waiter.resolve(actionData);
            }
            catch (err) {
                logger.warn({ action: actionData.action, err }, "Waiter resolve threw");
            }
            return;
        }
        // Otherwise buffer (drop oldest if overflow)
        if (this._queue.length >= MAX_QUEUE_SIZE) {
            const dropped = this._queue.shift();
            logger.warn({ droppedAction: dropped?.action, queueSize: MAX_QUEUE_SIZE }, "SSE queue overflow — dropping oldest event");
        }
        this._queue.push(actionData);
    }
    /** Get next event, with optional timeout in ms. Returns null on timeout or close. */
    async getEvent(timeoutMs) {
        // Drain buffer first
        if (this._queue.length > 0) {
            return this._queue.shift();
        }
        if (this._closed)
            return null;
        const deferred = createDeferred();
        this._waiters.push(deferred);
        if (timeoutMs === undefined) {
            return deferred.promise;
        }
        const timer = setTimeout(() => {
            const idx = this._waiters.indexOf(deferred);
            if (idx >= 0) {
                this._waiters.splice(idx, 1);
                deferred.resolve(null);
            }
        }, timeoutMs);
        try {
            const result = await deferred.promise;
            clearTimeout(timer);
            return result;
        }
        catch {
            clearTimeout(timer);
            return null;
        }
    }
    close() {
        this._closed = true;
        // Resolve all pending waiters with null
        for (const waiter of this._waiters) {
            waiter.resolve(null);
        }
        this._waiters = [];
    }
    // ===== Convenience Methods =====
    emitAgentActivate(agentName, agentId, message = "") {
        this.emit({
            action: Action.activate_agent,
            agent_name: agentName ?? this._agentName ?? "Agent",
            agent_id: agentId,
            process_task_id: this._taskId,
            message,
        });
    }
    emitAgentDeactivate(agentName, agentId, message = "", tokensUsed) {
        const durationSeconds = this._startTime
            ? (Date.now() - this._startTime) / 1000
            : undefined;
        this.emit({
            action: Action.deactivate_agent,
            agent_name: agentName ?? this._agentName ?? "Agent",
            agent_id: agentId,
            process_task_id: this._taskId,
            message,
            tokens_used: tokensUsed,
            duration_seconds: durationSeconds,
        });
    }
    emitToolkitActivate(toolkitName, methodName, inputPreview, message = "", agentName) {
        this.emit({
            action: Action.activate_toolkit,
            toolkit_name: toolkitName,
            method_name: methodName,
            agent_name: agentName ?? this._agentName,
            process_task_id: this._taskId,
            input_preview: inputPreview?.slice(0, 200),
            message,
        });
    }
    emitToolkitDeactivate(toolkitName, methodName, outputPreview, success = true, durationMs, message = "", agentName) {
        this.emit({
            action: Action.deactivate_toolkit,
            toolkit_name: toolkitName,
            method_name: methodName,
            agent_name: agentName ?? this._agentName,
            process_task_id: this._taskId,
            output_preview: outputPreview?.slice(0, 200),
            success,
            duration_ms: durationMs,
            message,
        });
    }
    emitAgentThinking(content, agentName, step) {
        this.emit({
            action: Action.agent_thinking,
            agent_name: agentName ?? this._agentName ?? "Agent",
            thinking: content,
            step,
        });
    }
    emitTerminal(command, output, exitCode, workingDirectory, durationMs) {
        this.emit({
            action: Action.terminal,
            command,
            output: output?.slice(0, 2000),
            exit_code: exitCode,
            working_directory: workingDirectory,
            duration_ms: durationMs,
        });
    }
    emitBrowserAction(actionType, target, value, success = true, pageUrl, pageTitle, screenshotUrl, webviewId) {
        this.emit({
            action: Action.browser_action,
            action_type: actionType,
            target,
            value,
            success,
            page_url: pageUrl,
            page_title: pageTitle,
            screenshot_url: screenshotUrl,
            webview_id: webviewId,
        });
    }
    emitScreenshot(screenshot, url, pageTitle, tabId, webviewId) {
        this.emit({
            action: Action.screenshot,
            screenshot,
            url,
            page_title: pageTitle,
            tab_id: tabId,
            webview_id: webviewId,
        });
    }
    emitWriteFile(filePath, fileName, fileSize, contentPreview, mimeType) {
        this.emit({
            action: Action.write_file,
            file_path: filePath,
            file_name: fileName,
            file_size: fileSize,
            content_preview: contentPreview,
            mime_type: mimeType,
        });
    }
    emitNotice(title, message, level = "info", durationMs) {
        this.emit({
            action: Action.notice,
            level,
            title,
            message,
            duration_ms: durationMs,
        });
    }
    emitError(error, errorType, recoverable = true, details) {
        this.emit({
            action: Action.error,
            error,
            error_type: errorType,
            recoverable,
            details,
        });
    }
    emitHeartbeat() {
        this.emit({
            action: Action.heartbeat,
            message: "keep-alive",
        });
    }
    emitAgentReport(message, reportType = "info", agentType, executorId, taskLabel, subtaskLabel) {
        this.emit({
            action: Action.agent_report,
            message,
            report_type: reportType,
            agent_type: agentType,
            executor_id: executorId,
            task_label: taskLabel,
            subtask_label: subtaskLabel,
        });
    }
    emitEnd(status, message, result) {
        this.emit({
            action: Action.end,
            status,
            message,
            result,
        });
    }
    emitWaitConfirm(content, question, context = "initial", attachments, executorId, taskLabel) {
        this.emit({
            action: Action.wait_confirm,
            content,
            question,
            context,
            attachments,
            executor_id: executorId,
            task_label: taskLabel,
        });
    }
    emitTaskDecomposed(subtasks, summaryTask, originalTaskId) {
        this.emit({
            action: Action.task_decomposed,
            subtasks,
            summary_task: summaryTask,
            original_task_id: originalTaskId,
            total_subtasks: subtasks.length,
        });
    }
    emitSubtaskState(subtaskId, state, result, failureCount = 0, executorId, taskLabel) {
        this.emit({
            action: Action.subtask_state,
            subtask_id: subtaskId,
            state,
            result,
            failure_count: failureCount,
            executor_id: executorId,
            task_label: taskLabel,
        });
    }
    emitTaskReplanned(subtasks, originalTaskId, reason) {
        this.emit({
            action: Action.task_replanned,
            subtasks,
            original_task_id: originalTaskId,
            reason,
        });
    }
    emitWorkerAssigned(workerName, subtaskId, subtaskContent, workerId, executorId, taskLabel) {
        this.emit({
            action: Action.worker_assigned,
            worker_name: workerName,
            worker_id: workerId,
            subtask_id: subtaskId,
            subtask_content: subtaskContent,
            executor_id: executorId,
            task_label: taskLabel,
        });
    }
    emitWorkerCompleted(workerName, subtaskId, resultPreview, durationSeconds, workerId, executorId, taskLabel) {
        this.emit({
            action: Action.worker_completed,
            worker_name: workerName,
            worker_id: workerId,
            subtask_id: subtaskId,
            result_preview: resultPreview,
            duration_seconds: durationSeconds,
            executor_id: executorId,
            task_label: taskLabel,
        });
    }
    emitWorkerFailed(workerName, subtaskId, error, failureCount = 0, willRetry = false, workerId, executorId, taskLabel) {
        this.emit({
            action: Action.worker_failed,
            worker_name: workerName,
            worker_id: workerId,
            subtask_id: subtaskId,
            error,
            failure_count: failureCount,
            will_retry: willRetry,
            executor_id: executorId,
            task_label: taskLabel,
        });
    }
    emitMemoryResult(pathsCount, paths, hasWorkflow = false, method) {
        this.emit({
            action: Action.memory_result,
            paths_count: pathsCount,
            paths,
            has_workflow: hasWorkflow,
            method,
        });
    }
    emitStepStarted(stepIndex, stepName, stepDescription) {
        this.emit({
            action: Action.step_started,
            step_index: stepIndex,
            step_name: stepName,
            step_description: stepDescription,
        });
    }
    emitStepCompleted(stepIndex, stepName, result, durationSeconds) {
        this.emit({
            action: Action.step_completed,
            step_index: stepIndex,
            step_name: stepName,
            result: result?.slice(0, 500),
            duration_seconds: durationSeconds,
        });
    }
    emitStepFailed(stepIndex, stepName, error, recoverable = true) {
        this.emit({
            action: Action.step_failed,
            step_index: stepIndex,
            step_name: stepName,
            error,
            recoverable,
        });
    }
}
//# sourceMappingURL=emitter.js.map