/**
 * AMI Task Executor — Parallel batch execution with dependency resolution.
 *
 * Ported from ami_task_executor.py.
 *
 * Key features:
 * - Parallel dispatch: eligible subtasks (all deps satisfied) run concurrently
 * - Semaphore-based concurrency limit (MAX_PARALLEL_SUBTASKS)
 * - Per-subtask browser sessions for parallel browser agents
 * - workflow_guide injected as explicit instruction in prompt
 * - SSE events for real-time UI updates
 * - Pause/resume support
 * - Fail-fast: if a subtask fails, skip all dependents
 * - Replan tool injection per subtask
 */
import type { AgentTool } from "@mariozechner/pi-agent-core";
import { type AMISubtask, type ExecutionResult, type ReplanResult, type TaskExecutorLike, type AgentLike } from "./schemas.js";
import type { SSEEmitter } from "../events/emitter.js";
export declare class AMITaskExecutor implements TaskExecutorLike {
    readonly taskId: string;
    readonly executorId: string;
    readonly taskLabel: string;
    private emitter?;
    private apiKey?;
    private userRequest;
    private maxRetries;
    private maxTurnsPerSubtask;
    private userId?;
    private agentTools;
    private systemPrompts;
    private workspaceDir;
    private childAgentToolsFactory?;
    private _subtasks;
    private subtaskMap;
    private _paused;
    private pauseResolvers;
    private _stopped;
    private _runningAgents;
    private _sessionPool;
    constructor(opts: {
        taskId: string;
        emitter?: SSEEmitter;
        apiKey?: string;
        agentTools: Map<string, AgentTool<any>[]>;
        systemPrompts: Map<string, string>;
        maxRetries?: number;
        maxTurnsPerSubtask?: number;
        userRequest?: string;
        executorId?: string;
        taskLabel?: string;
        userId?: string;
        workspaceDir?: string;
        childAgentToolsFactory?: (agentType: string, sessionId: string, workingDirOverride?: string) => AgentTool<any>[];
    });
    get isPaused(): boolean;
    get subtasks(): AMISubtask[];
    getCurrentAgent(): AgentLike | null;
    getRunningAgents(): Map<string, AgentLike>;
    setSubtasks(subtasks: AMISubtask[]): void;
    stop(): void;
    pause(): void;
    resume(): void;
    execute(): Promise<ExecutionResult>;
    private executeBatch;
    private getAllEligibleSubtasks;
    private _borrowSessionId;
    private _returnSessionId;
    private _closeAllPooledSessions;
    private executeSubtask;
    private buildPrompt;
    /**
     * Save large subtask result to a file in workspace.
     * Returns the filename (agent cwd is already workspace).
     */
    private saveResultToFile;
    /**
     * List files in workspace for prompt injection.
     * Returns a compact file listing so the agent knows what data
     * is available from earlier tasks without an extra tool call.
     */
    private getWorkspaceListing;
    replanSubtasks(newSubtasks: AMISubtask[]): ReplanResult;
    addSubtasksAsync(newSubtasks: AMISubtask[], afterSubtaskId?: string): Promise<string[]>;
    /**
     * Remove all dynamic subtasks spawned by a parent subtask.
     * Called before retrying a failed subtask to prevent duplicate
     * dynamic subtasks from being re-added on retry.
     */
    private removeDynamicSubtasks;
    private getSubtaskProgress;
    private classifyError;
    private emitSubtaskState;
    private emitSubtaskRunning;
    private emitNewlyFailed;
    /**
     * Check if post-execution learning should be triggered.
     *
     * Conditions (matching Python _should_trigger_learning):
     * - Execution was not stopped/cancelled
     * - User session is active (daemon has auth tokens)
     * - At least 1 browser subtask
     * - Total subtask count >= 2
     * - All browser subtasks succeeded
     */
    private shouldTriggerLearning;
    /**
     * Fire-and-forget: send execution data to Cloud Backend for learning.
     */
    private learnFromExecution;
    /**
     * Start BehaviorRecorder for a browser subtask.
     * Returns the recorder instance, or null if startup fails.
     * All errors are caught — recorder failure must not block task execution.
     */
    private startBehaviorRecorder;
    /**
     * Stop a running BehaviorRecorder.
     */
    private stopBehaviorRecorder;
    /**
     * Save recorded operations to Memory via CloudClient.
     * Only called when a subtask succeeds (SubtaskState.DONE).
     */
    private saveRecordedOperations;
    private waitIfPaused;
}
//# sourceMappingURL=task-executor.d.ts.map