/**
 * TaskState — per-task state management.
 *
 * Manages event queue, human response queue, user message queue,
 * conversation history, and execution metadata.
 */
import { SSEEmitter } from "../events/emitter.js";
// ===== Task Status =====
export var TaskStatus;
(function (TaskStatus) {
    TaskStatus["PENDING"] = "pending";
    TaskStatus["RUNNING"] = "running";
    TaskStatus["WAITING"] = "waiting";
    TaskStatus["COMPLETED"] = "completed";
    TaskStatus["FAILED"] = "failed";
    TaskStatus["CANCELLED"] = "cancelled";
})(TaskStatus || (TaskStatus = {}));
function createDeferred() {
    let resolve;
    const promise = new Promise((res) => {
        resolve = res;
    });
    return { resolve, promise };
}
// ===== TaskState =====
const MAX_CONVERSATION_BYTES = 100 * 1024; // 100KB
export class TaskState {
    // Identity
    taskId;
    task;
    userId;
    projectId;
    // Status
    status = TaskStatus.PENDING;
    progress = 0;
    result;
    error;
    // Timestamps
    createdAt = new Date();
    startedAt;
    completedAt;
    updatedAt = new Date();
    // Tool tracking
    toolsCalled = [];
    loopIteration = 0;
    // Conversation history
    conversationHistory = [];
    lastTaskResult;
    // Subtask tracking
    subtasks = [];
    summaryTask;
    // SSE emitter for streaming events
    emitter;
    // Abort controller for cancellation
    abortController = new AbortController();
    // Human response queue (for ask_human tool)
    _humanResponseQueue = [];
    // User message queue (for multi-turn conversation)
    _userMessageQueue = [];
    _userMessageWaiters = [];
    // Pause support
    _pauseDeferred;
    constructor(taskId, task) {
        this.taskId = taskId;
        this.task = task;
        this.emitter = new SSEEmitter();
        this.emitter.configure(taskId);
    }
    // ===== Status transitions =====
    markRunning() {
        this.status = TaskStatus.RUNNING;
        this.startedAt = new Date();
        this.updatedAt = new Date();
    }
    markWaiting() {
        this.status = TaskStatus.WAITING;
        this.updatedAt = new Date();
    }
    markCompleted(result) {
        this.status = TaskStatus.COMPLETED;
        this.result = result;
        this.completedAt = new Date();
        this.updatedAt = new Date();
    }
    markFailed(error) {
        this.status = TaskStatus.FAILED;
        this.error = error;
        this.completedAt = new Date();
        this.updatedAt = new Date();
    }
    markCancelled(reason) {
        this.status = TaskStatus.CANCELLED;
        this.error = reason;
        this.completedAt = new Date();
        this.updatedAt = new Date();
        this.abortController.abort();
        // Resolve pause deferred so executor doesn't hang
        if (this._pauseDeferred) {
            this._pauseDeferred.resolve();
            this._pauseDeferred = undefined;
        }
        // Drain all pending waiters so nothing hangs
        this._drainWaiters();
    }
    /** Resolve all pending deferred queues (used on cancel/cleanup). */
    _drainWaiters() {
        // Drain user message waiters — resolve with null so getUserMessage() returns null
        for (const waiter of this._userMessageWaiters) {
            waiter.resolve(null);
        }
        this._userMessageWaiters = [];
        // Drain human response waiters — resolve with null so waitForHumanResponse() returns null
        for (const waiter of this._humanResponseQueue) {
            waiter.resolve(null);
        }
        this._humanResponseQueue = [];
    }
    // ===== Conversation history =====
    addConversation(role, content) {
        this.conversationHistory.push({
            role,
            content,
            timestamp: new Date().toISOString(),
        });
        // Auto-trim if too large
        let totalSize = this.conversationHistory.reduce((sum, e) => sum + e.content.length, 0);
        while (totalSize > MAX_CONVERSATION_BYTES && this.conversationHistory.length > 2) {
            const removed = this.conversationHistory.shift();
            totalSize -= removed.content.length;
        }
    }
    getRecentContext(maxEntries = 10) {
        const recent = this.conversationHistory.slice(-maxEntries);
        return recent
            .map((e) => `[${e.role}]: ${e.content}`)
            .join("\n\n");
    }
    // ===== Human response queue =====
    /** Wait for a human response. Resolves when provideHumanResponse() is called. */
    waitForHumanResponse(timeoutMs = 300_000) {
        const deferred = createDeferred();
        this._humanResponseQueue.push(deferred);
        const timer = setTimeout(() => {
            // Remove stale deferred so provideHumanResponse doesn't resolve a dead consumer
            const idx = this._humanResponseQueue.indexOf(deferred);
            if (idx >= 0)
                this._humanResponseQueue.splice(idx, 1);
        }, timeoutMs);
        return Promise.race([
            deferred.promise.then((v) => { clearTimeout(timer); return v; }),
            new Promise((resolve) => setTimeout(() => resolve(null), timeoutMs)),
        ]);
    }
    /** Provide a human response to the waiting ask_human tool. */
    provideHumanResponse(response) {
        const waiter = this._humanResponseQueue.shift();
        if (waiter) {
            waiter.resolve(response);
            return true;
        }
        return false;
    }
    // ===== User message queue =====
    /** Put a user message into the queue for multi-turn processing. */
    putUserMessage(message) {
        // If someone is waiting, resolve immediately
        if (this._userMessageWaiters.length > 0) {
            const waiter = this._userMessageWaiters.shift();
            waiter.resolve(message);
            return;
        }
        this._userMessageQueue.push(message);
    }
    /** Get the next user message, with optional timeout. Returns null on timeout or cancel. */
    getUserMessage(timeoutMs) {
        // Drain buffer first
        if (this._userMessageQueue.length > 0) {
            return Promise.resolve(this._userMessageQueue.shift());
        }
        const deferred = createDeferred();
        this._userMessageWaiters.push(deferred);
        if (timeoutMs === undefined) {
            return deferred.promise;
        }
        const timer = setTimeout(() => {
            // Remove stale deferred so putUserMessage doesn't resolve a dead consumer
            const idx = this._userMessageWaiters.indexOf(deferred);
            if (idx >= 0)
                this._userMessageWaiters.splice(idx, 1);
        }, timeoutMs);
        return Promise.race([
            deferred.promise.then((v) => { clearTimeout(timer); return v; }),
            new Promise((resolve) => setTimeout(() => resolve(null), timeoutMs)),
        ]);
    }
    /**
     * Cancel the most recent getUserMessage() waiter.
     * Used when an executor wins the Promise.race in waitForEvent() —
     * the losing getUserMessage() deferred must be removed so it doesn't
     * consume the next putUserMessage() call.
     */
    cancelLastGetUserMessage() {
        const waiter = this._userMessageWaiters.pop();
        if (waiter) {
            waiter.resolve(null); // resolve the orphaned promise to prevent leaks
        }
    }
    /** Check if there are pending user messages without consuming them. */
    get hasPendingUserMessages() {
        return this._userMessageQueue.length > 0;
    }
    // ===== Pause/Resume =====
    /** Pause execution. Returns a promise that resolves when resume() is called. */
    async pause() {
        this.status = TaskStatus.WAITING;
        this.updatedAt = new Date();
        this._pauseDeferred = createDeferred();
        return this._pauseDeferred.promise;
    }
    /** Resume paused execution. No-op if already cancelled/completed/failed. */
    resume() {
        if (this.status === TaskStatus.CANCELLED ||
            this.status === TaskStatus.COMPLETED ||
            this.status === TaskStatus.FAILED) {
            return false;
        }
        if (this._pauseDeferred) {
            this.status = TaskStatus.RUNNING;
            this.updatedAt = new Date();
            this._pauseDeferred.resolve();
            this._pauseDeferred = undefined;
            return true;
        }
        return false;
    }
    get isPaused() {
        return this._pauseDeferred !== undefined;
    }
    // ===== Duration =====
    get durationSeconds() {
        if (!this.startedAt)
            return undefined;
        const end = this.completedAt ?? new Date();
        return (end.getTime() - this.startedAt.getTime()) / 1000;
    }
    // ===== Serialization =====
    toJSON() {
        return {
            task_id: this.taskId,
            task: this.task,
            status: this.status,
            progress: this.progress,
            result: this.result,
            error: this.error,
            created_at: this.createdAt.toISOString(),
            started_at: this.startedAt?.toISOString(),
            completed_at: this.completedAt?.toISOString(),
            user_id: this.userId,
            project_id: this.projectId,
            loop_iterations: this.loopIteration,
            tools_called_count: this.toolsCalled.length,
            has_result: this.result !== undefined,
            has_error: this.error !== undefined,
            subtasks: this.subtasks,
            summary_task: this.summaryTask,
        };
    }
}
//# sourceMappingURL=task-state.js.map