/**
 * Cloud Client — HTTP client for AMI cloud backend API.
 *
 * Ported from CloudClient in Python daemon.
 *
 * Endpoints: check_version, memory/plan, memory/query, memory/learn,
 * memory/add, memory/stats, memory/phrases, recordings, intent-builder, etc.
 */
import { createLogger } from "../utils/logging.js";
import { getConfig } from "../utils/config.js";
import { getAuthToken } from "./auth-manager.js";
const logger = createLogger("cloud-client");
export class CloudClient {
    baseUrl;
    timeout = 30_000;
    constructor(opts) {
        this.baseUrl = opts?.baseUrl ?? getConfig().cloud.api_url;
        logger.info({ baseUrl: this.baseUrl }, "CloudClient initialized");
    }
    // ===== Generic Request =====
    async request(method, path, body, extraHeaders, creds, timeoutMs) {
        const url = `${this.baseUrl}${path}`;
        const headers = {
            "Content-Type": "application/json",
            ...extraHeaders,
        };
        // Resolve token: explicit creds > daemon-managed token
        const usingDaemonToken = !creds?.token;
        let token = creds?.token;
        if (!token) {
            token = (await getAuthToken()) ?? undefined;
        }
        if (token) {
            headers["Authorization"] = `Bearer ${token}`;
        }
        logger.debug({ method, path }, "Cloud API request");
        const resp = await fetch(url, {
            method,
            headers,
            body: body ? JSON.stringify(body) : undefined,
            signal: AbortSignal.timeout(timeoutMs ?? this.timeout),
        });
        // Auto-retry on 401 when using daemon-managed token (token may have been refreshed)
        if (resp.status === 401 && usingDaemonToken) {
            logger.info({ method, path }, "Got 401, attempting token refresh and retry");
            const freshToken = await getAuthToken();
            if (freshToken && freshToken !== token) {
                headers["Authorization"] = `Bearer ${freshToken}`;
                const retryResp = await fetch(url, {
                    method,
                    headers,
                    body: body ? JSON.stringify(body) : undefined,
                    signal: AbortSignal.timeout(timeoutMs ?? this.timeout),
                });
                if (retryResp.ok) {
                    const ct = retryResp.headers.get("content-type") ?? "";
                    return ct.includes("application/json") ? retryResp.json() : retryResp.text();
                }
                const text = await retryResp.text().catch(() => "");
                throw new Error(`Cloud API error ${retryResp.status} ${method} ${path}: ${text}`);
            }
        }
        if (!resp.ok) {
            const text = await resp.text().catch(() => "");
            throw new Error(`Cloud API error ${resp.status} ${method} ${path}: ${text}`);
        }
        const contentType = resp.headers.get("content-type") ?? "";
        if (contentType.includes("application/json")) {
            return resp.json();
        }
        return resp.text();
    }
    get(path, creds, extraHeaders) {
        return this.request("GET", path, undefined, extraHeaders, creds);
    }
    post(path, body, creds, extraHeaders) {
        return this.request("POST", path, body, extraHeaders, creds);
    }
    del(path, creds, extraHeaders) {
        return this.request("DELETE", path, undefined, extraHeaders, creds);
    }
    patch(path, body, creds, extraHeaders) {
        return this.request("PATCH", path, body, extraHeaders, creds);
    }
    // ===== Version =====
    async checkVersion(clientVersion, platform, creds) {
        try {
            const result = (await this.post("/api/v1/app/version-check", { version: clientVersion, platform }, creds));
            return result;
        }
        catch (err) {
            logger.warn({ err }, "Version check failed");
            return { compatible: true };
        }
    }
    // ===== Memory API =====
    async memoryQuery(body, creds) {
        return this.post("/api/v1/memory/query", body, creds);
    }
    async memoryPlan(body, creds) {
        // PlannerAgent runs LLM agent loop — needs longer timeout than default 30s
        return this.request("POST", "/api/v1/memory/plan", body, undefined, creds, 120_000);
    }
    async memoryLearn(body, creds) {
        // LearnerAgent runs LLM agent loop with tool calls — typically 20-50s, can exceed 60s for large tasks
        return this.request("POST", "/api/v1/memory/learn", body, undefined, creds, 120_000);
    }
    async memoryAdd(body, creds) {
        return this.post("/api/v1/memory/add", body, creds);
    }
    async memoryStats(creds) {
        return this.get("/api/v1/memory/stats", creds);
    }
    async memoryDelete(creds) {
        return this.del("/api/v1/memory", creds);
    }
    // ===== CognitivePhrase API =====
    async listPhrases(limit = 50, creds) {
        return this.get(`/api/v1/memory/phrases?limit=${limit}`, creds);
    }
    async listPublicPhrases(limit = 50, sort = "popular", creds) {
        return this.get(`/api/v1/memory/phrases/public?limit=${limit}&sort=${sort}`, creds);
    }
    async getPhrase(phraseId, source, creds) {
        const query = source ? `?source=${source}` : "";
        return this.get(`/api/v1/memory/phrases/${phraseId}${query}`, creds);
    }
    async deletePhrase(phraseId, creds) {
        return this.del(`/api/v1/memory/phrases/${phraseId}`, creds);
    }
    async publishPhrase(phraseId, creds) {
        return this.post("/api/v1/memory/share", { phrase_id: phraseId }, creds);
    }
    async unpublishPhrase(phraseId, creds) {
        return this.post("/api/v1/memory/unpublish", { phrase_id: phraseId }, creds);
    }
    async getPublishStatus(phraseId, creds) {
        return this.get(`/api/v1/memory/publish-status?phrase_id=${encodeURIComponent(phraseId)}`, creds);
    }
    // ===== Recordings =====
    async getRecording(sessionId, creds) {
        return this.get(`/api/v1/recordings/${sessionId}`, creds);
    }
    async analyzeRecording(sessionId, userId, creds) {
        return this.post(`/api/v1/recordings/${sessionId}/analyze`, {
            user_id: userId,
        }, creds);
    }
    // ===== Intent Builder =====
    async createIntentBuilderSession(body, creds) {
        return this.post("/api/v1/intent-builder/sessions", body, creds);
    }
    async intentBuilderChat(sessionId, message, creds) {
        return this.post(`/api/v1/intent-builder/sessions/${sessionId}/chat`, {
            message,
        }, creds);
    }
    async getIntentBuilderState(sessionId, creds) {
        return this.get(`/api/v1/intent-builder/sessions/${sessionId}/state`, creds);
    }
    async deleteIntentBuilderSession(sessionId, creds) {
        return this.del(`/api/v1/intent-builder/sessions/${sessionId}`, creds);
    }
    /**
     * Stream SSE from intent builder session.
     * Returns raw fetch Response for SSE streaming.
     */
    async intentBuilderStream(sessionId, creds) {
        const url = `${this.baseUrl}/api/v1/intent-builder/sessions/${sessionId}/stream`;
        const headers = {};
        // Resolve token: explicit creds > daemon-managed token
        const token = creds?.token ?? (await getAuthToken());
        if (token)
            headers["Authorization"] = `Bearer ${token}`;
        // SSE streams are long-lived — use 10 min timeout instead of the default 30s
        return fetch(url, {
            headers,
            signal: AbortSignal.timeout(600_000),
        });
    }
}
// ===== Singleton =====
let _client = null;
export function getCloudClient() {
    if (!_client) {
        _client = new CloudClient();
    }
    return _client;
}
export function setCloudClient(client) {
    _client = client;
}
//# sourceMappingURL=cloud-client.js.map