/**
 * Auth Manager — Central JWT token management for the daemon.
 *
 * Single source of truth for auth state. Persists session to ~/.ami/session.json.
 * Auto-refreshes tokens before expiry. Thread-safe via refresh mutex.
 *
 * Exports:
 * - getAuthToken()   — returns valid access_token, auto-refreshes if expiring soon
 * - storeSession()   — persist tokens + user info to file
 * - clearSession()   — delete session file
 * - getSession()     — returns user info (no tokens exposed)
 * - hasSession()     — quick check if session exists
 */
export interface SessionInfo {
    logged_in: boolean;
    username?: string;
    user_id?: string;
    email?: string;
}
/**
 * Get a valid access token. Auto-refreshes if expiring within 5 minutes.
 * Returns null if no session or refresh fails.
 */
export declare function getAuthToken(): Promise<string | null>;
/**
 * Store a new session (called after login/register).
 */
export declare function storeSession(data: {
    access_token: string;
    refresh_token: string;
    username: string;
    user_id: string;
    email?: string;
}): void;
/**
 * Update tokens in an existing session (called after token refresh).
 * Only overwrites fields that are provided; preserves everything else.
 */
export declare function updateSessionTokens(data: {
    access_token: string;
    refresh_token?: string;
}): void;
/**
 * Clear the stored session (logout).
 */
export declare function clearSession(): void;
/**
 * Get public session info (no tokens exposed).
 */
export declare function getSession(): SessionInfo;
/**
 * Quick check if a session exists.
 */
export declare function hasSession(): boolean;
//# sourceMappingURL=auth-manager.d.ts.map