/**
 * Memory Routes — Proxy to cloud backend memory API.
 *
 * POST   /api/v1/memory/add
 * POST   /api/v1/memory/query
 * GET    /api/v1/memory/stats
 * DELETE /api/v1/memory
 * GET    /api/v1/memory/phrases
 * GET    /api/v1/memory/phrases/public
 * GET    /api/v1/memory/phrases/:phraseId
 * DELETE /api/v1/memory/phrases/:phraseId
 * POST   /api/v1/memory/share
 * POST   /api/v1/memory/unpublish
 * GET    /api/v1/memory/publish-status
 */
import { Router } from "express";
import { getCloudClient } from "../services/cloud-client.js";
import { createLogger } from "../utils/logging.js";
const logger = createLogger("memory-routes");
export const memoryRouter = Router();
// ===== Helper: extract per-request credentials =====
function getCredentials(req) {
    const authHeader = req.headers["authorization"];
    const token = authHeader?.startsWith("Bearer ") ? authHeader.slice(7) : undefined;
    return { token };
}
/**
 * Extract HTTP status code from CloudClient error messages.
 * CloudClient throws: "Cloud API error 401 GET /path: ..."
 */
function getErrorStatus(err) {
    if (err instanceof Error) {
        const match = err.message.match(/Cloud API error (\d{3})/);
        if (match)
            return parseInt(match[1], 10);
    }
    return 500;
}
// ===== POST /add =====
memoryRouter.post("/add", async (req, res) => {
    try {
        const client = getCloudClient();
        const creds = getCredentials(req);
        const result = await client.memoryAdd(req.body, creds);
        res.json(result);
    }
    catch (err) {
        res.status(getErrorStatus(err)).json({ error: String(err) });
    }
});
// ===== POST /query =====
memoryRouter.post("/query", async (req, res) => {
    try {
        const client = getCloudClient();
        const creds = getCredentials(req);
        const result = await client.memoryQuery(req.body, creds);
        res.json(result);
    }
    catch (err) {
        res.status(getErrorStatus(err)).json({ error: String(err) });
    }
});
// ===== GET /stats =====
memoryRouter.get("/stats", async (req, res) => {
    try {
        const client = getCloudClient();
        const creds = getCredentials(req);
        const result = await client.memoryStats(creds);
        res.json(result);
    }
    catch (err) {
        res.status(getErrorStatus(err)).json({ error: String(err) });
    }
});
// ===== GET /debug =====
memoryRouter.get("/debug", (_req, res) => {
    res.json({
        success: false,
        error: "Debug not available in proxy mode",
    });
});
// ===== DELETE / =====
memoryRouter.delete("/", async (req, res) => {
    try {
        const client = getCloudClient();
        const creds = getCredentials(req);
        const result = await client.memoryDelete(creds);
        res.json(result);
    }
    catch (err) {
        res.status(getErrorStatus(err)).json({ error: String(err) });
    }
});
// ===== GET /phrases =====
memoryRouter.get("/phrases", async (req, res) => {
    try {
        const client = getCloudClient();
        const creds = getCredentials(req);
        const limit = parseInt(req.query.limit) || 50;
        const result = await client.listPhrases(limit, creds);
        res.json(result);
    }
    catch (err) {
        res.status(getErrorStatus(err)).json({ error: String(err) });
    }
});
// ===== GET /phrases/public =====
memoryRouter.get("/phrases/public", async (req, res) => {
    try {
        const client = getCloudClient();
        const creds = getCredentials(req);
        const limit = parseInt(req.query.limit) || 50;
        const sort = req.query.sort ?? "popular";
        const result = await client.listPublicPhrases(limit, sort, creds);
        res.json(result);
    }
    catch (err) {
        res.status(getErrorStatus(err)).json({ error: String(err) });
    }
});
// ===== GET /phrases/:phraseId =====
memoryRouter.get("/phrases/:phraseId", async (req, res) => {
    try {
        const client = getCloudClient();
        const creds = getCredentials(req);
        const source = req.query.source;
        const result = await client.getPhrase(req.params.phraseId, source, creds);
        res.json(result);
    }
    catch (err) {
        res.status(getErrorStatus(err)).json({ error: String(err) });
    }
});
// ===== DELETE /phrases/:phraseId =====
memoryRouter.delete("/phrases/:phraseId", async (req, res) => {
    try {
        const client = getCloudClient();
        const creds = getCredentials(req);
        const result = await client.deletePhrase(req.params.phraseId, creds);
        res.json(result);
    }
    catch (err) {
        res.status(getErrorStatus(err)).json({ error: String(err) });
    }
});
// ===== POST /share =====
memoryRouter.post("/share", async (req, res) => {
    try {
        const client = getCloudClient();
        const creds = getCredentials(req);
        const { phrase_id } = req.body;
        const result = await client.publishPhrase(phrase_id, creds);
        res.json(result);
    }
    catch (err) {
        res.status(getErrorStatus(err)).json({ error: String(err) });
    }
});
// ===== POST /unpublish =====
memoryRouter.post("/unpublish", async (req, res) => {
    try {
        const client = getCloudClient();
        const creds = getCredentials(req);
        const { phrase_id } = req.body;
        const result = await client.unpublishPhrase(phrase_id, creds);
        res.json(result);
    }
    catch (err) {
        res.status(getErrorStatus(err)).json({ error: String(err) });
    }
});
// ===== GET /publish-status =====
memoryRouter.get("/publish-status", async (req, res) => {
    try {
        const phraseId = req.query.phrase_id;
        if (!phraseId) {
            res.status(400).json({ error: "phrase_id query parameter required" });
            return;
        }
        const client = getCloudClient();
        const creds = getCredentials(req);
        const result = await client.getPublishStatus(phraseId, creds);
        res.json(result);
    }
    catch (err) {
        res.status(getErrorStatus(err)).json({ error: String(err) });
    }
});
//# sourceMappingURL=memory.js.map