/**
 * Intent Builder Routes — Proxy to cloud backend intent builder API.
 *
 * POST   /api/v1/intent-builder/sessions               — create session
 * GET    /api/v1/intent-builder/sessions/:id/stream     — SSE stream
 * POST   /api/v1/intent-builder/sessions/:id/chat       — send message
 * GET    /api/v1/intent-builder/sessions/:id/state      — get state
 * DELETE /api/v1/intent-builder/sessions/:id            — delete session
 */
import { Router } from "express";
import { getCloudClient } from "../services/cloud-client.js";
import { createLogger } from "../utils/logging.js";
const logger = createLogger("intent-builder-routes");
export const intentBuilderRouter = Router();
function getCredentials(req) {
    const authHeader = req.headers["authorization"];
    const token = authHeader?.startsWith("Bearer ") ? authHeader.slice(7) : undefined;
    return { token };
}
/**
 * Extract HTTP status code from CloudClient error messages.
 * CloudClient throws: "Cloud API error 401 GET /path: ..."
 */
function getErrorStatus(err) {
    if (err instanceof Error) {
        const match = err.message.match(/Cloud API error (\d{3})/);
        if (match)
            return parseInt(match[1], 10);
    }
    return 500;
}
// ===== POST /sessions =====
intentBuilderRouter.post("/sessions", async (req, res) => {
    try {
        const client = getCloudClient();
        const creds = getCredentials(req);
        const result = await client.createIntentBuilderSession(req.body, creds);
        res.json(result);
    }
    catch (err) {
        res.status(getErrorStatus(err)).json({ error: String(err) });
    }
});
// ===== GET /sessions/:sessionId/stream =====
intentBuilderRouter.get("/sessions/:sessionId/stream", async (req, res) => {
    const { sessionId } = req.params;
    const creds = getCredentials(req);
    try {
        const client = getCloudClient();
        const upstream = await client.intentBuilderStream(sessionId, creds);
        if (!upstream.ok) {
            res.status(upstream.status).json({
                error: `Upstream error: ${upstream.status}`,
            });
            return;
        }
        // Set SSE headers
        res.writeHead(200, {
            "Content-Type": "text/event-stream",
            "Cache-Control": "no-cache",
            "Connection": "keep-alive",
            "X-Accel-Buffering": "no",
        });
        // Pipe upstream SSE to client
        const reader = upstream.body?.getReader();
        if (!reader) {
            res.end();
            return;
        }
        // Cancel upstream reader on client disconnect
        req.on("close", () => {
            reader.cancel().catch(() => { });
        });
        const decoder = new TextDecoder();
        try {
            while (true) {
                const { done, value } = await reader.read();
                if (done)
                    break;
                const text = decoder.decode(value, { stream: true });
                res.write(text);
            }
        }
        catch {
            // Stream closed
        }
        finally {
            reader.cancel().catch(() => { });
            res.end();
        }
    }
    catch (err) {
        if (!res.headersSent) {
            res.status(500).json({ error: String(err) });
        }
        else {
            res.end();
        }
    }
});
// ===== POST /sessions/:sessionId/chat =====
intentBuilderRouter.post("/sessions/:sessionId/chat", async (req, res) => {
    const { message } = req.body;
    if (!message || typeof message !== "string") {
        res.status(400).json({ error: "message field is required" });
        return;
    }
    try {
        const client = getCloudClient();
        const creds = getCredentials(req);
        const result = await client.intentBuilderChat(req.params.sessionId, message, creds);
        res.json(result);
    }
    catch (err) {
        res.status(getErrorStatus(err)).json({ error: String(err) });
    }
});
// ===== GET /sessions/:sessionId/state =====
intentBuilderRouter.get("/sessions/:sessionId/state", async (req, res) => {
    try {
        const client = getCloudClient();
        const creds = getCredentials(req);
        const result = await client.getIntentBuilderState(req.params.sessionId, creds);
        res.json(result);
    }
    catch (err) {
        res.status(getErrorStatus(err)).json({ error: String(err) });
    }
});
// ===== DELETE /sessions/:sessionId =====
intentBuilderRouter.delete("/sessions/:sessionId", async (req, res) => {
    try {
        const client = getCloudClient();
        const creds = getCredentials(req);
        const result = await client.deleteIntentBuilderSession(req.params.sessionId, creds);
        res.json(result);
    }
    catch (err) {
        res.status(getErrorStatus(err)).json({ error: String(err) });
    }
});
//# sourceMappingURL=intent-builder.js.map