/**
 * BrowserSession — CDP connection to Electron's embedded Chromium.
 *
 * Ported from browser_session.py (HybridBrowserSession).
 *
 * Key concepts:
 * - Connects via CDP to Electron's Chromium (BROWSER_CDP_PORT env var)
 * - 8 WebContentsView pool (shared `persist:user_login` partition)
 * - Pool pages marked with `ami=pool` URL
 * - Tab management with Tab Group support
 * - Singleton per session-id
 */
import { type Page } from "playwright";
import { PageSnapshot } from "./page-snapshot.js";
import { ActionExecutor } from "./action-executor.js";
interface TabGroup {
    taskId: string;
    title: string;
    color: string;
    tabs: Map<string, Page>;
    currentTabId?: string;
}
export declare class BrowserSession {
    private static _instances;
    private static _daemonSession;
    private _browser;
    private _context;
    private _playwright;
    private _pages;
    private _page;
    private _currentTabId;
    private static _claimedPages;
    private static _poolClaimLock;
    private _tabGroups;
    private _colorIndex;
    private _pageToViewId;
    snapshot: PageSnapshot | null;
    executor: ActionExecutor | null;
    private _sessionId;
    private _cdpPort;
    constructor(sessionId: string);
    get sessionId(): string;
    get isConnected(): boolean;
    get currentPage(): Page | null;
    get currentTabId(): string | null;
    /** Return the Electron WebContentsView ID for the active page ("0"-"7" or null). */
    get webviewId(): string | null;
    static getInstance(sessionId: string): BrowserSession;
    /** Get an existing instance without creating a new one. */
    static getExistingInstance(sessionId: string): BrowserSession | null;
    static getDaemonSession(): BrowserSession | null;
    ensureBrowser(): Promise<void>;
    private _initializeFromPool;
    /**
     * Atomically find and claim ONE free pool page.
     * Returns null if no pool pages are available (non-fatal for init).
     * Only the scan + claim + goto("about:blank") need the lock.
     */
    private _takeOnePoolPage;
    /**
     * Register a claimed pool page as this session's active page.
     * Sets up snapshot, executor, and event listeners.
     *
     * No setViewportSize needed — Electron's WebContentsView.setBounds()
     * controls the CSS layout viewport directly. Off-screen views use
     * 1920×1080 bounds so the agent gets a full viewport.
     */
    private _registerClaimedPage;
    private _handleNewPage;
    private _claimPoolPage;
    private _returnPageToPool;
    getPage(): Promise<Page>;
    visit(url: string): Promise<string>;
    getSnapshot(options?: {
        forceRefresh?: boolean;
        diffOnly?: boolean;
        viewportLimit?: boolean;
    }): Promise<string>;
    getSnapshotWithElements(options?: {
        viewportLimit?: boolean;
    }): Promise<Record<string, unknown>>;
    execAction(action: Record<string, unknown>): Promise<Record<string, unknown>>;
    getTabInfo(): Promise<Record<string, unknown>[]>;
    switchToTab(tabId: string): Promise<boolean>;
    closeTab(tabId: string): Promise<boolean>;
    createNewTab(url?: string): Promise<[string, Page]>;
    createTabGroup(taskId: string, title?: string): Promise<TabGroup>;
    createTabInGroup(taskId: string, url?: string): Promise<[string, Page]>;
    closeTabGroup(taskId: string): Promise<boolean>;
    getTabGroupsInfo(): Record<string, unknown>[];
    takeScreenshot(): Promise<string | null>;
    static startDaemonSession(): Promise<BrowserSession>;
    static stopDaemonSession(): Promise<void>;
    close(): Promise<void>;
    static closeAllSessions(): Promise<void>;
}
export {};
//# sourceMappingURL=browser-session.d.ts.map